package nl.b3p.formendpoint.controller;

import io.swagger.v3.oas.annotations.media.Schema;
import nl.b3p.formendpoint.resource.Feature;
import org.hibernate.Session;
import org.hibernate.type.StandardBasicTypes;
import org.locationtech.jts.geom.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.repository.support.JpaRepositoryFactory;
import org.springframework.web.bind.annotation.RestController;

import javax.persistence.EntityManager;
import javax.persistence.OneToMany;
import javax.persistence.Query;
import javax.sql.DataSource;
import java.lang.reflect.Field;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

@RestController
public class UserLayerController implements FormController<Feature> {

    Logger logger = LoggerFactory.getLogger(this.getClass());

    public final static String USERLAYER_SEPARATOR = "ul_";
    private String layerId;

    private DataSource dataSource;
    private EntityManager em;

    private FormController origController;

    private String userLayerFilter = null;
    private String userLayerTable = null;
    private String userLayerOriginalTable = null;

    private String query;

    public UserLayerController() {

    }

    public UserLayerController(String layerId, DataSource ds, Map<String, FormController> controllers, EntityManager em) throws Exception {
        this.layerId = layerId;
        this.dataSource = ds;
        this.em = em;
        this.init(controllers);
        if (this.origController == null) {
            throw new IllegalArgumentException("Userlayer controller incomplete data: original controller not found for " + this.userLayerOriginalTable);
        }
        this.query = "select " + getColumns() + " FROM " + this.userLayerTable;
    }

    private String getColumns() {
        List<String> columns = new ArrayList<>();
        Field[] fields = this.origController.getMaintainingClass().getDeclaredFields();

        for (Field f : fields) {
            if (Geometry.class.isAssignableFrom(f.getType())) {
                columns.add("st_curvetoline(" + f.getName() + ") as " + f.getName());
            } else {
                OneToMany otm = f.getAnnotation(OneToMany.class);
                if (otm == null) {
                    String name = f.getName();
                    Schema s = f.getAnnotation(Schema.class);
                    if (s != null) {
                        name = s.name();
                    }
                    columns.add(name);
                }
            }
        }

        return String.join(",", columns);
    }

    private void init(Map<String, FormController> controllers) {
        try (Connection con = this.dataSource.getConnection();) {

            String queryDetails = "select layer, value, details_key from layer_details where layer = ?";
            try (PreparedStatement stmt = con.prepareStatement(queryDetails)) {
                stmt.setLong(1, Long.parseLong(layerId));
                ResultSet rs = stmt.executeQuery();
                while (rs.next()) {
                    String key = rs.getString("details_key");
                    String value = rs.getString("value");
                    switch (key) {
                        case "userlayer_filter":
                            userLayerFilter = value;
                            break;
                        case "userlayer_original_layername":
                            userLayerOriginalTable = value;
                            break;
                    }
                }
            } catch (SQLException e) {
                logger.error("Error retrieving userlayer details: ", e);
            }

            String queryLayer = "select id, name from layer where id = ?";

            try (PreparedStatement stmt = con.prepareStatement(queryLayer)) {
                stmt.setLong(1, Long.parseLong(layerId));
                ResultSet rs = stmt.executeQuery();
                while (rs.next()) {
                    userLayerTable = rs.getString("name");
                }
            } catch (SQLException e) {
                logger.error("Error retrieving layername: ", e);
            }

        } catch (SQLException e) {
            logger.error("Error opening connection: ", e);
        }

        this.origController = controllers.get(sanitizeLayerName(this.userLayerOriginalTable));
    }

    private String sanitizeLayerName(String layerName) {
        if(layerName.contains(":")){
            layerName = layerName.substring(layerName.indexOf(":") + 1);
        }
        return layerName.replace("gb_", "");
    }

    @Override
    public String getMaintainingClassString() {
        return "userlayer";
    }

    @Override
    public Class getMaintainingClass() {
        return null;
    }

    @Override
    public Feature get(String objectGuid) {
        String sql = query + " WHERE  object_guid = :objectguid";
        Query q = this.em.createNativeQuery(sql, this.origController.getMaintainingClass());
        q.setParameter("objectguid", objectGuid);
        Feature l = (Feature) q.getSingleResult();
        return l;
    }

    @Override
    public List<Feature> onPoint(double x, double y, double scale) {
        GeometryFactory gf = new GeometryFactory(new PrecisionModel(2), 28992);
        Point p = gf.createPoint(new Coordinate(x, y));
        Geometry geom = p.buffer(scale);

        String sql = query + " WHERE  st_intersects(geometrie, :area) = true";
        Query q = this.em.createNativeQuery(sql, this.origController.getMaintainingClass());
        q.setParameter("area", geom);
        List<Feature> l = q.getResultList();
        return l;
    }


    @Override
    public Page<Feature> getAllPaged(Pageable pageable) {

        logger.error("Cannot save userlayer");
        return null;
    }

    @Override
    public List<Feature> getAll() {
        String sql = query;
        Query q = this.em.createNativeQuery(sql, this.origController.getMaintainingClass());
        List<Feature> l = q.getResultList();
        return l;
    }

    @Override
    public Feature save(Feature feature, String parentId) {
        logger.error("Cannot save userlayer");
        return null;
    }

    @Override
    public Feature update(String objectGuid, Feature feature) {
        logger.error("Cannot update userlayer");
        return null;
    }

    @Override
    public void delete(String objectGuid) {
        logger.error("Cannot delete userlayer");
    }

}
