package nl.b3p.formendpoint.controller;

import com.fasterxml.jackson.annotation.JsonSubTypes;
import nl.b3p.formendpoint.resource.Feature;
import nl.b3p.formendpoint.resource.FeaturetypeMetadata;
import nl.b3p.formendpoint.resource.GeometryType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;

import javax.annotation.PostConstruct;
import javax.persistence.EntityManager;
import javax.sql.DataSource;
import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.*;

import static nl.b3p.formendpoint.controller.UserLayerController.USERLAYER_SEPARATOR;

@RestController
@RequestMapping("features")
public class FeatureController implements FormController{
    Logger logger = LoggerFactory.getLogger(FeatureController.class);
    Map<String, FormController> controllers = new HashMap<>();

    @Autowired
    @Qualifier("tailormapDB")
    private DataSource dataSource;

    @Autowired
    private EntityManager em;

    @Autowired
    private List<FormController> ll;

    @PostConstruct
    public void init(){
        logger.debug("Adding formcontrollers to FeatureController...");
        Map<String, String> englishToDutch = new HashMap<>();

        String totalAdded = "";
        JsonSubTypes a = Feature.class.getAnnotation(JsonSubTypes.class);
        JsonSubTypes.Type[] ts = a.value();

        for (JsonSubTypes.Type type: ts) {
            englishToDutch.put(type.value().getSimpleName().toLowerCase(), type.name().toLowerCase());
        }
        englishToDutch.put("userlayer", "userlayer");
        for (FormController fc:ll) {
            if(!(fc instanceof FeatureController)){

                controllers.put(englishToDutch.get(fc.getMaintainingClassString()), fc);
                totalAdded += fc.getMaintainingClassString() + ", ";
                logger.info("Added: " + fc.getMaintainingClassString() + " - " + fc.getClass().getSimpleName());
            }
        }
        logger.debug( "Added: " + totalAdded);
    }

    @GetMapping("/{featuretype}/{objectGuid}")
    public Feature get(@PathVariable String featuretype, @PathVariable String objectGuid) throws Exception {
        FormController fc = getController(featuretype);
        return fc.get(objectGuid);
    }

    @GetMapping(value = "/info/{featureTypes}")
    public List<FeaturetypeMetadata> featuretypeInformation(@PathVariable String[] featureTypes) {
        List<FeaturetypeMetadata> md = new ArrayList<>();

        controllers.values().forEach(controller -> {
            if(! (controller instanceof UserLayerController)) {
                int b = 0;
                md.add(getMetadata(controller));
            }
        });
        return md;
    }

    private FeaturetypeMetadata getMetadata(FormController c) {
        FeaturetypeMetadata fm = new FeaturetypeMetadata();
        Type[] genericInterfaces = c.getClass().getGenericInterfaces();
        Type t = genericInterfaces[0];
        Type[] types = ((ParameterizedType)t).getActualTypeArguments();
        Type resource = types[0];
        try {
            Feature f =(Feature) ((Class) resource).newInstance();
            fm.featuretypeName = f.getClazz();
        } catch (InstantiationException e) {
            e.printStackTrace();
        } catch (IllegalAccessException e) {
            e.printStackTrace();
        }
        try {
            Field f = ((Class) resource).getDeclaredField("geometrie");
            fm.geometryAttribute = f.getName();
            fm.geometryType = GeometryType.fromValue(f.getType().getSimpleName()).orElse(GeometryType.POLYGON);
        }catch(NoSuchFieldException e){

        }
        return fm;
    }

    @GetMapping(value = "/unpaged")
    public List<Feature> getAll(){
        List<Feature> fs= new ArrayList<>();
        controllers.values().forEach(controller ->{
            fs.addAll(controller.getAll());
        });
        return fs;
    }

    @GetMapping(value = "/{x}/{y}/{scale}")
    public List<Feature> onPoint(@PathVariable double x, @PathVariable double y, @PathVariable double scale){
        List<Feature> features = new ArrayList<>();

        controllers.values().forEach(controller ->{
            features.addAll(controller.onPoint(x, y, scale));
        });

        return features;
    }

   @GetMapping(value = "/{featureTypes}/{x}/{y}/{scale}")
   public List<Feature> featuretypeOnPoint(@PathVariable String[] featureTypes, @PathVariable double x,
                                           @PathVariable double y, @PathVariable double scale)  throws Exception {
       List<Feature> features = new ArrayList<>();
       List<FormController> fcs = getControllers(featureTypes);
       fcs.forEach(fc -> {
           features.addAll(fc.onPoint(x, y, scale));
       });
       return features;
   }

    @PostMapping
    public Feature save(@RequestBody Feature f, @RequestParam(required = false) String parentId) throws Exception {
        FormController fc = getController(f);
        f.setObjectGuid(UUID.randomUUID().toString());
        return fc.save(f, parentId);
    }

    @Override
    @PutMapping("/{objectGuid}")
    public Feature update(@PathVariable  String objectGuid, @RequestBody Feature feature) throws Exception {
        FormController fc = getController(feature);
        return fc.update(objectGuid, feature);
    }

    @Transactional
    @DeleteMapping("{featuretype}/{objectGuid}")
    public void delete(@PathVariable String featuretype, @PathVariable String objectGuid) throws Exception {
        FormController fc = getController(featuretype);
        fc.delete(objectGuid);
    }

    private FormController getController(Feature f) throws Exception {
        return getController(f.getClazz());
    }

    private FormController getController(String featuretype) throws Exception {
        FormController fc = null;
        if(featuretype.contains(USERLAYER_SEPARATOR)){
            fc = getUserLayerController(featuretype);
        }else{
            fc = controllers.get(featuretype);
        }
        return fc;
    }

    private List<FormController> getControllers(String[] featuretypes) throws Exception {
        List<FormController> fcs = new ArrayList<>();
        for (String ft: featuretypes) {
            if(controllers.containsKey(ft)){
                fcs.add(controllers.get(ft));
            }
        }
        fcs.addAll(getUserLayerControllers(featuretypes));
        return fcs;
    }

    private List<UserLayerController> getUserLayerControllers(String[] featureTypes) throws Exception {
        List<UserLayerController> fcs = new ArrayList<>();
        for (String ft : featureTypes) {
            if(ft.contains(USERLAYER_SEPARATOR)){
                fcs.add(getUserLayerController(ft));
            }
        }
        return fcs;
    }

    private UserLayerController getUserLayerController(String ft) throws Exception {
        String layerId = ft.substring(USERLAYER_SEPARATOR.length());
        UserLayerController ulc = new UserLayerController(layerId, this.dataSource, this.controllers, this.em);
        return ulc;
    }

    @Override
    public Class getMaintainingClass() {
        return Feature.class;
    }

    @Override
    public Feature get(String objectGuid) {
        // not useful when not knowing the featuretype
        throw new IllegalArgumentException("Not intented to be implemented");
    }

    @Override
    public Page getAllPaged(Pageable pageable) {
        throw new IllegalArgumentException("Not intented to be implemented");
    }

    @Override
    public void delete(String objectGuid) {
        throw new IllegalArgumentException("Not intented to be implemented");
    }
}
