



/*
 * The Unified Mapping Platform (JUMP) is an extensible, interactive GUI 
 * for visualizing and manipulating spatial features with geometry and attributes.
 *
 * Copyright (C) 2003 Vivid Solutions
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * For more information, contact:
 *
 * Vivid Solutions
 * Suite #1A
 * 2328 Government Street
 * Victoria BC  V8T 5G5
 * Canada
 *
 * (250)385-6040
 * www.vividsolutions.com
 */

package org.locationtech.jts.wms;

import java.util.*;

import org.locationtech.jts.jump.util.CollectionUtil;

/**
 * Represents a WMS Layer.
 *
 * @author Chris Hodgson chodgson@refractions.net
 */
public class MapLayer {

  // immutable members
  private MapLayer parent;
  private String name;
  private String title;
  private ArrayList srsList;  
  private ArrayList subLayers;
  private BoundingBox bbox;
  
  // user modifiable members
  private boolean enabled = false;
  
  /**
   * Creates a new instance of MapLayer 
   */
  public MapLayer(String name, String title, Collection srsList, Collection subLayers, BoundingBox bbox) {
    this.parent = null;
    this.name = name;
    this.title = title;
    this.srsList = new ArrayList( srsList );
    this.subLayers = new ArrayList( subLayers );
    Iterator it = subLayers.iterator();
    while( it.hasNext() ) { 
      ((MapLayer)it.next()).parent = this;
    }
    this.bbox = bbox;
  }
  
  /**
   * Returns the number of sub-layers that this MapLayer has.
   * @return the number of sub-layers that this MapLayer has
   */
  public int numSubLayers() {
    return subLayers.size();
  }
  
  /**
   * Returns the sub-layer at the specified index.
   * @param n the index of the sub-layer to return
   * @return the MapLayer sub-layer at the specified index
   */
  public MapLayer getSubLayer( int n ) {
    return (MapLayer)subLayers.get( n );
  }

  /**
   * Gets a copy of the list of the sublayers of this layer.
   * @return a copy of the Arraylist containing all the sub-layers of this layer
   */
  public ArrayList getSubLayerList() {
    return (ArrayList)subLayers.clone();
  }

  /**
   * Returns a list of all the layers in order of a root-left-right traversal of
   * the layer tree.
   * @return a list of all the layers in order of a root-left-right traversal of
   * the layer tree.
   */
  public ArrayList getLayerList() {
    ArrayList list = new ArrayList();
    list.add( this );
    Iterator it = subLayers.iterator();
    while( it.hasNext() ) {
      list.addAll( ((MapLayer)it.next()).getLayerList() );
    }
    return list;
  }  

  /**
   * Gets the title of this MapLayer.
   * The title of a layer should be used for display purposes.
   * @return the title of this Layer
   */
  public String getTitle() {
    return title;
  }
  
  /**
   * Gets the name of this Layer.
   * The name of a layer is its 'back-end', ugly name, which generally 
   * shouldn't need to be used by others but is available anyway.
   * Layers which do not have any data associated with them, such as container
   * or grouping layers, might not have a name, in which case null will be
   * returned.
   * @return the name of the layer, or null if it doesn't have a name
   */
  public String getName() {
    return name;
  }
  
  /**
   * Gets the parent MapLayer of this MapLayer. 
   * @return the parent layer of this MapLayer, or null if the layer has no parent.
   */
  public MapLayer getParent() {
    return parent;
  }
  
  /**
   * Gets the LatLonBoundingBox for this layer.
   * If this layer doesn't have a LatLonBoundingBox specified, we recursively
   * ask the parent layer for its bounding box. The WMS spec says that each
   * layer should either have its own LatLonBoundingBox, or inherit one from
   * its parent, so this recursive call should be successful. If not, null is
   * returned. However, if a bounding box is returned, it will have the 
   * SRS string "LatLon". 
   * Note that the BoundingBox is not necessarily "tight".
   * @return the BoundingBox for this layer, or null if the BBox is unknown
   */
  public BoundingBox getBoundingBox() {
    if( bbox != null ) {
      return bbox;
    } 
    if( parent != null ) {
      return parent.getBoundingBox();
    }
    return null;
  }
  
  /**
   * Returns a copy of the list of supported SRS's. Each SRS is a string in the
   * format described by the WMS specification, such as "EPSG:1234".
   * @return a copy of the list of supported SRS's
   */
  public ArrayList getSRSList() {
    return (ArrayList)srsList.clone();
  }
  //<<TODO>>I'd like to return generic Lists, rather than concrete ArrayLists.
  //Or even better, Collections, since order is not significant (I think) [Jon Aquino]
  /**
   * @return a list of the SRS list of this MapLayer and its ancestors
   */
  public Collection getFullSRSList() {
    ArrayList fullSRSList  = new ArrayList(getSRSList());
    if (parent != null) fullSRSList.addAll(parent.getFullSRSList());
    return CollectionUtil.removeDuplicates(fullSRSList);
  }
  
  /**
   * Returns a somewhat nicely-formatted string representing all of the details of
   * this layer and its sub-layers (recursively).
   * @return a somewhat nicely-formatted string representing all of the details of
   * this layer and its sub-layers (recursively).
   */
  public String toString() {
    StringBuffer s = new StringBuffer( "WMSLayer {\n"
        + "  name: \"" + name + "\"\n"
        + "  title: \"" + title + "\"\n"
        + "  srsList: " + srsList.toString() + "\n"
        + "  subLayers: [\n" );
    for( int i = 0; i < subLayers.size(); i++ ) {
      s.append( subLayers.get( i ).toString() + ", " );
    }
    s.append( "  ]\n  bbox: " );
    if( bbox != null ) {
      s.append( bbox.toString() );
    } else {
      s.append( "null" );
    }
    s.append( "\n}\n" );
    return s.toString();
  }
}
