/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package nl.b3p.pzh.rwbp.parse;

import com.vividsolutions.jts.geom.Geometry;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import javax.persistence.EntityManager;
import nl.b3p.commons.services.StreamCopy;
import nl.b3p.pzh.rwbp.entity.Bouwplan;
import nl.b3p.pzh.rwbp.entity.BouwplanVariable;
import nl.b3p.pzh.rwbp.entity.Gebruiker;
import nl.b3p.pzh.rwbp.entity.Gebruikgegevens;
import nl.b3p.pzh.rwbp.entity.Gemeente;
import nl.b3p.pzh.rwbp.entity.Ontwikkelaartype;
import nl.b3p.pzh.rwbp.entity.Plantype;
import nl.b3p.pzh.rwbp.entity.Statusplanologisch;
import nl.b3p.pzh.rwbp.entity.Statusproject;
import nl.b3p.pzh.rwbp.export.ExportConfigServlet;
import nl.b3p.pzh.rwbp.stripes.BouwplanUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.geotools.data.DataStore;
import org.geotools.data.DataStoreFinder;
import org.geotools.data.FeatureSource;
import org.geotools.data.shapefile.ShapefileDataStoreFactory;
import org.geotools.feature.FeatureCollection;
import org.geotools.feature.FeatureIterator;
import org.opengis.feature.Feature;
import org.opengis.feature.Property;
import org.stripesstuff.stripersist.Stripersist;

/**
 *
 * @author Roy Braam
 * @author Meine Toonen
 */
public class ShapeParser extends Parser{
    private static final Log log = LogFactory.getLog(ShapeParser.class);
    public static final String ID = "id";
    public static final String NAAM = "plannaam";
    public static final String GEMEENTE = "gemeente";
    public static final String PLAATSNAAM = "PLAATSNAAM";
    public static final String REFERENTIE = "referentie";
    public static final String BESTEMMINGSPLANNAAM = "best-plan";
    public static final String IMRO_NUMMER = "imronummer";
    public static final String MASTERPLAN = "masterplan";
    public static final String ONTWIKKELAARSTYPE = "opdr_type";
    public static final String ONTWIKKELAAR_NAMELIJK = "opdr_naam";
    public static final String PLANTYPE = "plantype";
    public static final String STATUSPLAN = "status_pla";
    public static final String STATUSPROJECT = "status_pro";
    public static final String KNELPUNTEN = "knelpunten";
    public static final String JAAR_BEGIN = "jaar_begin";
    public static final String JAAR_EIND = "jaar_eind";
    public static final String PLANCAPACITEIT = "plancap";
    public static final String SLOOP = "sloop";

    public List<Bouwplan> parse(InputStream is, Gebruiker gebruiker) throws IOException, Exception {
        
        //write zip to working dir.
        String workingName = ExportConfigServlet.uniqueName(File.separator, "", true);
        File workingDir = new File(workingName);
        workingDir.mkdir();
        List<Bouwplan> bouwplannen=null;
        try{
            writeZipToWorkingDir(is,workingDir);
            bouwplannen= parseFiles(workingDir, gebruiker);
        }finally{
            deleteDir(workingDir);
        }
        
        return bouwplannen;
    }
    /**
     * Delete dir
     * @param dir The directory to delete
     * @return  If succeeded
     */
    public static boolean deleteDir(File dir) {
        if (dir.isDirectory()) {
            File[] children = dir.listFiles();
            for (int i=0; i<children.length; i++) {
                boolean success=false;
                if (children[i].isDirectory()){
                    deleteDir(children[i]);                    
                }else{
                    children[i].delete();
                }
            }
            return dir.delete();
        }else{
            return dir.delete();
        }
        // The directory is now empty so delete it
        //return dir.delete();
    }

    private void writeZipToWorkingDir(InputStream is, File workingDir) throws IOException {        
        ZipInputStream zip = new ZipInputStream(is);
        try{
            ZipEntry currentFile;
            byte[] buffer = new byte[2048];
            int length;
            while((currentFile =zip.getNextEntry())!=null){
                String newFile= workingDir.getAbsolutePath()+File.separator+currentFile.getName();                
                File file = new File(newFile);
                createParentDir(new File(newFile));
                if (currentFile.isDirectory()){
                    continue;
                }
                try{
                    OutputStream os = new FileOutputStream(newFile);
                    try{                              
                        StreamCopy.copy(zip, os);
                    }finally{
                        os.close();
                    } 
                }catch(FileNotFoundException fne){
                    log.debug(fne);
                }finally{
                    zip.closeEntry();
                }
            }
        }finally{
            zip.close();
        }
    }
    /**
     * Parse files to bouwplannen
     * @param the file.
     * return the bouwplannen
     */
    private List<Bouwplan> parseFiles(File file, Gebruiker gebruiker) throws IOException, Exception {
        List<Bouwplan> bouwplannen = new ArrayList<Bouwplan>();
        if (file.isDirectory() && 
                file.listFiles()!=null){            
            for (File f : file.listFiles()){
                bouwplannen.addAll(parseFiles(f, gebruiker));
            }
        }else if (file.isFile() &&
                file.getName().toLowerCase().endsWith(".shp")) {            
            bouwplannen.addAll(parseShape(file, gebruiker));
        }
        return bouwplannen;
            
    }
    /**
     * Parse shape file.
     * @param file the .shp file.
     * @return bouwplannen
     */
    private List<Bouwplan> parseShape(File file,Gebruiker gebruiker) throws IOException, Exception {
        Map map = new HashMap();
        map.put(ShapefileDataStoreFactory.URLP.key,file.toURI().toURL());
        
        DataStore datastore = DataStoreFinder.getDataStore(map);
        FeatureSource fs = datastore.getFeatureSource(datastore.getTypeNames()[0]);
        
        FeatureCollection fc = fs.getFeatures();
        FeatureIterator iterator = (FeatureIterator) fc.features();
        
        EntityManager em = Stripersist.getEntityManager();
        
        List<Bouwplan> bouwplannen = new ArrayList<Bouwplan>();
        try{
            while (iterator.hasNext()){
                EasyFeature f = new EasyFeature(iterator.next());
                Bouwplan b = new Bouwplan();
             
                //ID
                //NAAM
                //GEMEENTE
                //PLAATSNAAM
                //REFERENTIE
                //BESTEMMINGSPLANNAAM
                //IMRO_NUMMER
                //MASTERPLAN
                //ONTWIKKELAARSTYPE
                //ONTWIKKELAAR_NAMELIJK

                b.setNaam(f.getString(NAAM));
                String gemeente = f.getString(GEMEENTE);
                if (gemeente!=null){
                    List<Gemeente> list = em.createQuery("FROM Gemeente where naam=:n").setParameter("n", gemeente).getResultList();
                    if (list.size()>0){
                        b.setGemeente(list.get(0));
                    }
                }
                b.setPlaatsnaam(f.getString(PLAATSNAAM));
                b.setReferentienummer(f.getString(REFERENTIE));
                b.setBestemmingsplannaam(f.getString(BESTEMMINGSPLANNAAM));
                b.setImronummer(f.getString(IMRO_NUMMER));
                b.setMasterplan(f.getString(MASTERPLAN));
                //ontwikkelaar
                String ontwType = f.getString(ONTWIKKELAARSTYPE);
                if (ontwType!=null){
                    List<Ontwikkelaartype> list = em.createQuery("FROM Ontwikkelaartype where type=:t").setParameter("t",ontwType).getResultList();
                    if (list.size()>0){
                        b.setOntwikkelaartype(list.get(0));
                    }
                }else{
                    b.setOntwikkelaartype(null);
                }
                b.setOntwikkelaar_namelijk(f.getString(ONTWIKKELAAR_NAMELIJK));

                //PLANTYPE
                //STATUSPLAN
                //STATUSPROJECT
                String plantype = f.getString(PLANTYPE);
                if (plantype!=null){
                    List<Plantype> list = em.createQuery("FROM Plantype where type=:t").setParameter("t",plantype).getResultList();
                    if (list.size()>0){
                        b.setPlantype(list.get(0));
                    }
                }else{
                    b.setPlantype(null);
                }
                String statusPlan = f.getString(STATUSPLAN);

                if (statusPlan!=null){
                    List<Statusplanologisch> list = em.createQuery("FROM Statusplanologisch where type=:t").setParameter("t",statusPlan).getResultList();
                    if (list.size()>0){
                        b.setStatusplanologisch(list.get(0));
                    }
                }else{
                    b.setStatusplanologisch(null);
                }
                String statusProject = f.getString(STATUSPROJECT);

                if (statusProject!=null){
                    List<Statusproject> list = em.createQuery("FROM Statusproject where type=:t").setParameter("t",statusProject).getResultList();
                    if (list.size()>0){
                        b.setStatusproject(list.get(0));
                    }
                }

                if(b.getStatusproject() == null){
                    Statusproject sp = em.find(Statusproject.class,Statusproject.ONBEKEND_ID);
                    b.setStatusproject(sp);
                }

                 //KNELPUNTEN
                //JAAR_BEGIN
                //JAAR_EIND
                //PLANCAPACITEIT
                //SLOOP
                b.setKnelpunten(f.getString(KNELPUNTEN));
                b.setJaar_eerste_oplevering(f.getInteger(JAAR_BEGIN));
                b.setJaar_laatste_oplevering(f.getInteger(JAAR_EIND));
                b.setRestcapaciteit(f.getInteger(PLANCAPACITEIT));
                b.setResterendesloop(f.getInteger(SLOOP));

                Gebruikgegevens gg = em.find(Gebruikgegevens.class, BouwplanUtils.PRIVE);
                b.setGebruikgegevens(gg);
                
                b.setDatumlaatstewijziging(new Date());
                
                List<BouwplanVariable> bvs =loadPlanning(b,f);
                b.setBouwplanVariables(bvs);
                
                Geometry geom = f.getGeometry();  
                try{
                    b.setThe_geom(correctGeometry(geom));
                }catch (Exception cause) {
                    throw new Exception("Fout bij laden geometry van plan met naam: " + b.getNaam()+". Reden: "+cause.getMessage(), cause);
                }
                bouwplannen.add(b);
            }
        }finally{
            iterator.close();
        }
        return bouwplannen;
    }
   
    private List<BouwplanVariable> loadPlanning(Bouwplan b, EasyFeature f) throws IOException {
        List<BouwplanVariable> bvs = new ArrayList<BouwplanVariable>();

        for (int i = 2013; i < 2041; i++) {
            BouwplanVariable bv = getBouwplanVariable(b, f, i);
            bvs.add(bv);
        }
        return bvs;
    }

    private BouwplanVariable getBouwplanVariable(Bouwplan b, EasyFeature f, int year){
        Integer value = 0 ;
        String key = "gep" + year;
        BouwplanVariable var = new BouwplanVariable();
        try{
            value = f.getInteger(key);
        }catch (IOException ex){
        }
        var.setBouwplan(b);
        var.setGepland(value);
        var.setYear(year);
        return var;
    }

    private void createParentDir(File f) {
        if (!f.getParentFile().exists()){
            f.getParentFile().mkdir();
            createParentDir(f.getParentFile());
        }else{
            return;
        }
    }
    
    private class EasyFeature{
        private Feature feature;
        
        public EasyFeature(Feature f){
            this.feature=f;
        }
        
        public Feature getFeature(){
            return feature;
        }
        
        public Object getObject(String name){
            Property p =this.feature.getProperty(name);
            if (p==null){
                p = this.feature.getProperty(name.toUpperCase());            
                if (p == null){
                    p = this.feature.getProperty(name.toLowerCase());
                    if (p==null){
                        return null;
                    }
                }
            }
            
            return p.getValue();
        }
        
        public String getString(String name) throws IOException{
            Object o = getObject(name);
            if (o==null){
                return null;
            }
            if (o instanceof String){
                String s = (String)o;
                if (s.length() == 0){
                    return null;
                }else{
                    return (String) o;
                }
            }else{
                return o.toString();
            }
            //throw new IOException("Object \""+name+"\" not of type String: ");
        }

        public Date getDate(String name) throws IOException { 
            Object o = getObject(name);
            if (o==null){
                return null;
            }
            if (o instanceof Date){
                return (Date) o;
            }
            throw new IOException("Object \""+name+"\" not of type Date: ");
        }
        
        public Integer getInteger(String name) throws IOException { 
            Object o = getObject(name);
            if (o==null){
                return null;
            }
            if (o instanceof Integer){
                return (Integer) o;
            }else if (o instanceof Double){
                return ((Double)o).intValue();
            }else if (o instanceof Long){
                return ((Long)o).intValue();
            }
            throw new IOException("Object \""+name+"\" not of type Integer: ");
        }
        
        public Geometry getGeometry (){
            return (Geometry)this.feature.getDefaultGeometryProperty().getValue();
        }
    }
    
}

