package nl.b3p.pzh.rwbp.stripes;

import java.io.UnsupportedEncodingException;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.List;
import javax.persistence.EntityManager;
import net.sourceforge.stripes.action.Before;
import net.sourceforge.stripes.action.DefaultHandler;
import net.sourceforge.stripes.action.ForwardResolution;
import net.sourceforge.stripes.action.Resolution;
import net.sourceforge.stripes.action.SimpleMessage;
import net.sourceforge.stripes.action.StrictBinding;
import net.sourceforge.stripes.controller.LifecycleStage;
import net.sourceforge.stripes.tag.BeanFirstPopulationStrategy;
import net.sourceforge.stripes.validation.EmailTypeConverter;
import net.sourceforge.stripes.validation.LocalizableError;
import net.sourceforge.stripes.validation.SimpleError;
import net.sourceforge.stripes.validation.Validate;
import net.sourceforge.stripes.validation.ValidateNestedProperties;
import net.sourceforge.stripes.validation.ValidationErrors;
import net.sourceforge.stripes.validation.ValidationMethod;
import nl.b3p.commons.stripes.CustomPopulationStrategy;
import nl.b3p.pzh.rwbp.entity.Bouwplan;
import nl.b3p.pzh.rwbp.entity.Bouwplanopmerking;
import nl.b3p.pzh.rwbp.entity.Gebruiker;
import nl.b3p.pzh.rwbp.entity.Gemeente;
import nl.b3p.pzh.rwbp.entity.Provincie;
import nl.b3p.pzh.rwbp.entity.Regio;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.stripesstuff.stripersist.Stripersist;

/**
 *
 * @author Meine Toonen meinetoonen@b3partners.nl
 */
@StrictBinding
@CustomPopulationStrategy(BeanFirstPopulationStrategy.class)
public class AdminUserActionBean extends ViewerActionBean{
    private static final Log log = LogFactory.getLog(AdminUserActionBean.class);

    private Long exportAantal;

    private List<Gebruiker> gebruikersList = new ArrayList<Gebruiker>();
    private List<Provincie> provinciesList = new ArrayList<Provincie>();
    private List<Gemeente> gemeenteList = new ArrayList<Gemeente>();
    private List<Regio> regioList = new ArrayList<Regio>(); 
     
    @ValidateNestedProperties({
        @Validate(field="id", on="save"),
        @Validate(field="username", required=true, maxlength=255, on="save"),
        @Validate(field="fullname", required=true, maxlength=255, on="save"),
        @Validate(field="email", required=true, converter=EmailTypeConverter.class, maxlength=255, on="save"),
        @Validate(field="phone", maxlength=255, on="save")
    })
    private Gebruiker user;

    @Validate
    private boolean nieuweGebruiker = true;

    @Validate(maxlength=255)
    private String wachtwoord;
    @Validate(required=true, on="save")
    private String toegang;
    @Validate
    private Integer gemeente;
    @Validate
    private Integer regio;
    @Validate
    private Integer provincie;
    
    private final String JSP_USER = "/WEB-INF/jsp/admin/user.jsp";

    @DefaultHandler
    public Resolution view() {
        nieuweGebruiker = true;
        user = new Gebruiker();
        return new ForwardResolution(JSP_USER);
    }

    public Resolution selectGebruiker(){
        nieuweGebruiker = false;
        if (user!=null){
            if(user.getGemeente() != null){
                toegang = "gemeente";
            }else if(user.getRegio()!= null){
                toegang = "regio";
            }else if(user.getProvincie()!= null){
                toegang = "provincie";
            }else if(user.isRijk()){
                toegang = "rijk";
            }else if (user.isBeheerder()){
                toegang = "beheerder";
            }
        }

        return new ForwardResolution(JSP_USER);
    }

    @Before(stages = LifecycleStage.BindingAndValidation)
    public void loadEntity() {
        Integer gebruikerId = null;

        Integer id = null;
        try {
            gebruikerId = Integer.parseInt(context.getRequest().getParameter("gebruikerid"));
        } catch(NumberFormatException nfe) {
        }

        try {
            id = Integer.parseInt(context.getRequest().getParameter("user.id"));
        } catch(NumberFormatException nfe) {
        }

        if(id != null){
            gebruikerId = id;
        }

        if(gebruikerId != null) {
            EntityManager em = Stripersist.getEntityManager();

            user = (Gebruiker)em.find(Gebruiker.class, gebruikerId);
            if (!hasRightsToChange(gebruiker, user)){
                user=null;
                getContext().getMessages().add(new SimpleError("U heeft geen rechten om deze gebruiker aan te passen"));
            }
        }
    }

    public Resolution save() throws NoSuchAlgorithmException, UnsupportedEncodingException {
        EntityManager em = Stripersist.getEntityManager();

        if(wachtwoord != null){
            user.changePassword(context.getRequest(), wachtwoord);
        }
        ValidationErrors errors = context.getValidationErrors();
        if(nieuweGebruiker){
            if(toegang.equals("gemeente")){
                if (gemeente==null){
                    errors.add("gemeente", new LocalizableError("validation.required.valueNotPresent"));
                    return new ForwardResolution(JSP_USER);
                }
                Gemeente gem = em.find(Gemeente.class, gemeente);
                user.setGemeente(gem);
            }else if(toegang.equals("regio")){
                if (regio==null){
                    errors.add("regio", new LocalizableError("validation.required.valueNotPresent"));
                    return new ForwardResolution(JSP_USER);
                }
                Regio reg = em.find(Regio.class, regio);
                user.setRegio(reg);
            }else if(toegang.equals("provincie")){
                if (provincie==null){
                    errors.add("provincie", new LocalizableError("validation.required.valueNotPresent"));
                    return new ForwardResolution(JSP_USER);
                }
                Provincie p = (Provincie)em.find(Provincie.class, provincie);
                user.setProvincie(p);
            }else if( toegang.equals("beheerder")){
                user.setBeheerder(true);
            }else if( toegang.equals("rijk")){
                user.setRijk(true);
            }
        }
        if (!hasRightsToChange(gebruiker,user)){
            getContext().getMessages().add(new SimpleError("U heeft niet voldoende rechten om deze gebruiker te wijzigen."));
        }
        em.persist(user);
        em.flush();
        
        createLists();

        em.getTransaction().commit();
        
        SimpleMessage me = new SimpleMessage("Opslaan gelukt");
        context.getMessages().add(me);
        return new ForwardResolution(JSP_USER);
    }
    
    public Resolution delete(){
        if (user!=null){
            if (user.getId()== gebruiker.getId()){
                getContext().getMessages().add(new SimpleError("Het is niet mogelijk om de gebruiker te verwijderen waar u ook mee bent ingelogd"));
            }else{
                if(hasRightsToChange(gebruiker,user)){
                    EntityManager em = Stripersist.getEntityManager();

                    List<Bouwplan> bps = em.createQuery("FROM Bouwplan where gebruiker = :geb", Bouwplan.class).setParameter("geb", user).getResultList();
                    for (Bouwplan bp : bps) {
                        bp.setGebruiker(null);
                        em.persist(bp);
                    }
                    List<Bouwplanopmerking> opmerkingen = em.createQuery("FROM Bouwplanopmerking where gebruiker = :geb", Bouwplanopmerking.class).setParameter("geb", user).getResultList();
                    for (Bouwplanopmerking opmerking : opmerkingen) {
                        em.remove(opmerking);
                    }
                    
                    em.remove(user);                    
                    em.getTransaction().commit();
                    user=null;
                    createLists();
                }else{
                    getContext().getMessages().add(new SimpleError("Het is niet mogelijk om de gebruiker te verwijderen waar u ook mee bent ingelogd"));
                }
            }
        }
        return new ForwardResolution(JSP_USER);
    }

    @Before(stages = LifecycleStage.BindingAndValidation)
    private void createLists(){
        EntityManager em = Stripersist.getEntityManager();
 
        if(gebruiker.isBeheerder()){
            gemeenteList = em.createQuery("from Gemeente order by naam").getResultList();
        }else if (gebruiker.getProvincie() != null){ 
            gemeenteList = em.createQuery("from Gemeente WHERE provincie = :provincie order by naam").setParameter("provincie", gebruiker.getProvincie()).getResultList();
        }else{
            gemeenteList = new ArrayList<Gemeente>();
        }
        
        if(gebruiker.isBeheerder()){
            regioList = em.createQuery("from Regio order by naam").getResultList();
        }else if (gebruiker.getProvincie()!=null){
            regioList = em.createQuery("select r from Regio r join r.provincie p where p = :provincie order by r.naam").setParameter("provincie",gebruiker.getProvincie()).getResultList();
        }else{
            regioList = new ArrayList<Regio>();
        }
        
        if(gebruiker.isBeheerder()){
            gebruikersList = em.createQuery("from Gebruiker order by username").getResultList();
        }else if (gebruiker.isRijk()){
            gebruikersList = em.createQuery("from Gebruiker where rijk = :r and beheerder = false").setParameter("r",true).getResultList();
        }else if (gebruiker.getProvincie() != null){
            gebruikersList = em.createQuery("select g from Gebruiker g left join g.gemeente gem WHERE g.beheerder = false and (gem.provincie = :provincie OR g.provincie = :provincie OR g.regio in (select r from Regio r join r.provincie p where p = :provincie)) order by username")
                    .setParameter("provincie",gebruiker.getProvincie())
                    .getResultList();
        }
        
        if(gebruiker.isBeheerder()){
            provinciesList = em.createQuery("from Provincie order by naam").getResultList();
        }else if (gebruiker.getProvincie()!=null){
            provinciesList.add(gebruiker.getProvincie());
        }else{
            provinciesList = new ArrayList<Provincie>();
        }

        exportAantal = (Long)context.getRequest().getSession().getAttribute("exportAantal");
    }

    @ValidationMethod(on="save")
    public void validateGebruiker(){
        EntityManager em = Stripersist.getEntityManager();
        ValidationErrors errors = context.getValidationErrors();

        List users = null;
        if(!nieuweGebruiker){
            users = em.createQuery("from Gebruiker where username = :username and id != :id")
                .setParameter("username", user.getUsername()).setParameter("id", user.getId()).getResultList();
        }else{
            users = em.createQuery("from Gebruiker where username = :username")
                .setParameter("username", user.getUsername()).getResultList();
        }

        if(users != null && users.size() > 0){
            errors.add("gebruikersnaam", new LocalizableError("validation.field.notUnique"));
        }

        if(user.getUsername().equals(wachtwoord)){
            errors.add("gebruikersnaam en wachtwoord", new LocalizableError("validation.field.notTheSame"));
        }

        /*
         * Ook nog controleren of er wel een gemeente of regio is geselecteerd
         * als de gebruiker de daar bij horende rol krijgt.
         */
    }

    // <editor-fold defaultstate="collapsed" desc="getters en setters">
    public boolean isNieuweGebruiker() {
        return nieuweGebruiker;
    }

    public void setNieuweGebruiker(boolean nieuweGebruiker) {
        this.nieuweGebruiker = nieuweGebruiker;
    }

    public Integer getProvincie() {
        return provincie;
    }

    public void setProvincie(Integer provincie) {
        this.provincie = provincie;
    }

    public Long getExportAantal() {
        return exportAantal;
    }

    public void setExportAantal(Long exportAantal) {
        this.exportAantal = exportAantal;
    }

    public List<Gebruiker> getGebruikersList() {
        return gebruikersList;
    }

    public void setGebruikersList(List<Gebruiker> gebruikersList) {
        this.gebruikersList = gebruikersList;
    }
    
    public Integer getGemeente() {
        return gemeente;
    }

    public void setGemeente(Integer gemeente) {
        this.gemeente = gemeente;
    }

    public Integer getRegio() {
        return regio;
    }

    public void setRegio(Integer regio) {
        this.regio = regio;
    }

    public String getToegang() {
        return toegang;
    }

    public void setToegang(String toegang) {
        this.toegang = toegang;
    }

    public String getWachtwoord() {
        return wachtwoord;
    }

    public void setWachtwoord(String wachtwoord) {
        this.wachtwoord = wachtwoord;
    }

    public List<Provincie> getProvinciesList() {
        return provinciesList;
    }

    public void setProvinciesList(List<Provincie> provinciesList) {
        this.provinciesList = provinciesList;
    }

    public List<Gemeente> getGemeenteList() {
        return gemeenteList;
    }

    public void setGemeenteList(List<Gemeente> gemeenteList) {
        this.gemeenteList = gemeenteList;
    }

    public List<Regio> getRegioList() {
        return regioList;
    }

    public void setRegioList(List<Regio> regioList) {
        this.regioList = regioList;
    }

    public Gebruiker getUser() {
        return user;
    }

    public void setUser(Gebruiker user) {
        this.user = user;
    }
    // </editor-fold>

    private boolean hasRightsToChange(Gebruiker login, Gebruiker g) {
        if (login.isBeheerder()){
            return true;
        } 
        //Gebruiker is beheerder maar login niet >> mag niet. Alleen beheerder mag beheerders aanmaken.
        if (g.isBeheerder()){
            return false;
        }
        if (login.getProvincie()!=null){
            Provincie p=g.resolveProvincie();
            if (p!=null && p.getId().equals(login.getProvincie().getId())){
                return true;
            }
        }
        if (login.isRijk() && g.isRijk()){
            return true;
        }
        return false;
    }
}
