/*
 * Copyright (C) 2012-2015 B3Partners B.V.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package nl.b3p.pzh.rwbp.stripes;


import java.io.File;
import java.io.StringWriter;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import javax.persistence.EntityManager;
import javax.persistence.NoResultException;
import javax.persistence.Query;
import javax.servlet.http.HttpServletResponse;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.parsers.DocumentBuilderFactory;
import net.sourceforge.stripes.action.DefaultHandler;
import net.sourceforge.stripes.action.RedirectResolution;
import net.sourceforge.stripes.action.Resolution;
import net.sourceforge.stripes.action.StreamingResolution;
import net.sourceforge.stripes.action.StrictBinding;
import net.sourceforge.stripes.validation.OneToManyTypeConverter;
import net.sourceforge.stripes.validation.Validate;
import nl.b3p.imagetool.CombineImageSettings;
import nl.b3p.pzh.rwbp.entity.Bouwplan;
import nl.b3p.pzh.rwbp.entity.BouwplanAdditionalVariable;
import nl.b3p.pzh.rwbp.entity.BouwplanVariable;
import nl.b3p.pzh.rwbp.entity.ConfigurableVariable;
import nl.b3p.pzh.rwbp.entity.Gemeente;
import nl.b3p.pzh.rwbp.entity.Provincie;
import nl.b3p.pzh.rwbp.entity.ProvincieAdditionalVariable;
import nl.b3p.pzh.rwbp.entity.ProvincieAdditionalVariableValue;
import nl.b3p.pzh.rwbp.entity.ProvincieVariable;
import nl.b3p.pzh.rwbp.entity.Regio;
import nl.b3p.pzh.rwbp.entity.VariableType;
import nl.b3p.pzh.rwbp.printen.PrintExtraInfo;
import nl.b3p.pzh.rwbp.printen.PrintGenerator;
import nl.b3p.pzh.rwbp.printen.PrintInfo;
import nl.b3p.pzh.rwbp.printen.PrintInfoList;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.log4j.Logger;
import org.apache.xmlgraphics.util.MimeConstants;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.stripesstuff.stripersist.Stripersist;

/**
 *
 * @author Meine Toonen meinetoonen@b3partners.nl
 * @author Roy Braam roybraam@b3partners.nl
 * @author Eddy Scheper
 */
@StrictBinding
public class PrintActionBean extends ViewerActionBean {
    private static final Log log = LogFactory.getLog(PrintActionBean.class);
    protected static Logger fopLogger = Logger.getLogger("org.apache.fop");
    public static final String A5_Landscape = "A5_Landscape.xsl";
    public static final String A5_Portrait = "A5_Portrait.xsl";
    public static final String A4_Landscape = "A4_Landscape.xsl";
    public static final String A4_Portrait = "A4_Portrait.xsl";
    public static final String A3_Landscape = "A3_Landscape.xsl";
    public static final String A3_Portrait = "A3_Portrait.xsl";
    public static final String A0_Landscape = "A0_Landscape.xsl";
    public static final String A0_Portrait = "A0_Portrait.xsl";
    public static final String DEFAULT_TEMPLATE_PATH = "/WEB-INF/print/";
    public static final String A5 = "a5";
    public static final String A4 = "a4";
    public static final String A3 = "a3";
    public static final String A0 = "a0";
    public static final String LANDSCAPE = "landscape";
    public static final String PORTRAIT = "portrait";
    public static SimpleDateFormat df = new SimpleDateFormat("dd-MM-yyyy", new Locale("NL"));

    @Validate(converter = OneToManyTypeConverter.class,required = true)
    private List<Integer> bouwplans = new ArrayList<Integer>();
    

    @DefaultHandler
    public Resolution print() throws JSONException, Exception {
        boolean authorized = isAuthorized();
        if (!authorized) {
            return new RedirectResolution(SearchActionBean.class, "close");
        }
        //get the image url:
        final PrintInfoList pil = new PrintInfoList();
        final EntityManager em = Stripersist.getEntityManager();

        DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
        for (Integer id : bouwplans) {
            Bouwplan bouwplan = em.find(Bouwplan.class, id);
            CombineImageSettings cis = EditActionBean.createCombineImageSettings(bouwplan, this.getMapserverUrl(),this.getLayers(),1600 ,800,this.getHttpRequestParams());


            final PrintInfo info = new PrintInfo();
            if(cis != null){
                String imageUrl= getImageUrl(bouwplan, cis.getWidth(), cis.getHeight());
                info.setImageUrl(imageUrl +  "&key=" +cis.hashCode());
                info.setBbox(BouwplanUtils.makeBboxBouwplan(bouwplan));
            }

            info.setTitle(bouwplan.getNaam());

            info.setSubtitle("www.planmonitorwonen.nl");

            info.setDate(df.format(new Date()));

            info.setRemark("Extratesk");
            log.debug("Print Parse 'extra'");

            JSONArray jarray = getExtra(bouwplan);

            List<PrintExtraInfo> peis = new ArrayList<PrintExtraInfo>();
            for (int i = 0; i < jarray.length(); i++) {
                JSONObject extraObj = jarray.getJSONObject(i);
                PrintExtraInfo pei = new PrintExtraInfo();
                String className = extraObj.getString("className");
                String componentName = extraObj.getString("componentName");

                pei.setClassName(className);
                pei.setComponentName(componentName);
                pei.setInfoText(extraObj);
                peis.add(pei);
            }

            PrintExtraInfo additionalVars = new PrintExtraInfo();
            JSONObject vars = getAdditionalVars(bouwplan);

            additionalVars.setClassName("AdditionalVariables");
            additionalVars.setComponentName("AdditionalVariables");
            additionalVars.setInfoText(vars);
            peis.add(additionalVars);
            
            PrintExtraInfo subcategories = new PrintExtraInfo();
            JSONObject subVars = getSubcategories(bouwplan);
            subcategories.setClassName("SubCategories");
            subcategories.setComponentName("SubCategories");
            subcategories.setInfoText(subVars);
            peis.add(subcategories);
            
            PrintExtraInfo optionalVars = new PrintExtraInfo();
            JSONObject optVars = getOptionalVariables(bouwplan);
            optionalVars.setClassName("OptionalVariables");
            optionalVars.setComponentName("OptionalVariables");
            optionalVars.setInfoText(optVars);
            peis.add(optionalVars);
            
            PrintExtraInfo planningVars = new PrintExtraInfo();
            JSONObject dp = getDetailplannings(bouwplan);
            planningVars.setClassName("DetailPlanning");
            planningVars.setComponentName("DetailPlanning");
            planningVars.setInfoText(dp);
            peis.add(planningVars);

            info.setExtra(peis);
            pil.getInfos().add(info);
        }
        String pageFormat = A4;
        String orientation = PORTRAIT;
        final String templateName= getTemplateName(pageFormat,orientation);

        final String mimeType=MimeConstants.MIME_PDF;
        final String templateUrl =context.getServletContext().getRealPath(DEFAULT_TEMPLATE_PATH+templateName);;
        StreamingResolution res = new StreamingResolution(mimeType) {
            @Override
            public void stream(HttpServletResponse response) throws Exception {
                /* Set filename and extension */
                String filename = "Kaart_";
                if (bouwplans.size() == 1) {
                    Bouwplan bp = em.find(Bouwplan.class, bouwplans.get(0));
                    filename = bp.getNaam();
                }
                filename += pil.getInfos().get(0).getDate() + ".pdf";

                File f = new File(templateUrl);
                if (!f.exists()) {
                    f = new File(context.getServletContext().getRealPath(templateUrl));
                }
                if (!f.exists()) {
                    log.error("Can't find template: " + f.getAbsolutePath() + ". Using the default templates");
                    f = new File(context.getServletContext().getRealPath(DEFAULT_TEMPLATE_PATH + templateName));
                }
                try {

                    PrintGenerator.createOutput(pil, mimeType, f, true, response, filename);
                } finally {
                    for (PrintInfo info : pil.getInfos()) {

                        info.removeLegendImagesCache();
                    }
                }
            }
        };
        return res;
    }
    
    private JSONObject getDetailplannings(Bouwplan bouwplan) {
        JSONObject planning = new JSONObject();
        String detailType = bouwplan.getDetailType();
        planning.put("detailType", detailType);
        EntityManager em = Stripersist.getEntityManager();
        Query q = em.createQuery("FROM BouwplanVariable where bouwplan = :bouwplan order by year", BouwplanVariable.class).setParameter("bouwplan", bouwplan);
        List<BouwplanVariable> bvs = q.getResultList();
        JSONArray bvArray = new JSONArray();
        Map<Integer,Map<String, String>> bvsByYear = new HashMap<>();
        Set<String> allTypes = new HashSet<>();
        for (BouwplanVariable bv : bvs) {
            if(!bvsByYear.containsKey(bv.getYear())){
                bvsByYear.put(bv.getYear(), new HashMap<String, String>());
            }
            Map year = bvsByYear.get(bv.getYear());
            allTypes.add(bv.getType());
            Map map = bv.toPrintMap();
            year.putAll(map);
        }
        Map<String, List<String>> types = new HashMap<>();
        types.put(BouwplanVariable.TYPE_EENGEZINS, new ArrayList<String>(1));
        types.put(BouwplanVariable.TYPE_MEERGEZINS, new ArrayList<String>(1));
        types.put(BouwplanVariable.TYPE_ONBEKEND, new ArrayList<String>(1));
        for (String type : allTypes) {
            if(!types.containsKey(type)){
                String maincategory = type.substring(0, type.indexOf("-"));
                types.get(maincategory).add(type);
            }
            
        }
        
        Set<Integer> years = bvsByYear.keySet();
        for (Integer year : years) {
            Map bvMap = bvsByYear.get(year);
            JSONObject b = new JSONObject(bvMap);
            bvArray.put(b);
        }
        JSONObject typeInfo = new JSONObject();
        typeInfo.put("allTypes", allTypes);
        typeInfo.put("categories",types);
        
        if(detailType.equals("uitgebreid")){           
            Map<String, Map<String, Integer>> totalsMap = new HashMap<String, Map<String, Integer>>();
            
            for (BouwplanVariable bv : bvs) {
                String type = bv.getType();
                
                if(!totalsMap.containsKey(type)){
                    totalsMap.put(type, new HashMap<String, Integer>());
                    Map<String, Integer> totaltype = totalsMap.get(type);

                    totaltype.put("koop1", 0);
                    totaltype.put("koop2", 0);
                    totaltype.put("koop3", 0);
                    totaltype.put("koop4", 0);
                    totaltype.put("kooponb", 0);
                    totaltype.put("huur1", 0);
                    totaltype.put("huur2", 0);
                    totaltype.put("huur3", 0);
                    totaltype.put("huur4", 0);
                    totaltype.put("huuronb", 0);
                    totaltype.put("onbekend", 0);
                    totaltype.put("totaal", 0);
                    totaltype.put("subtotaal", 0);
                }

                Map<String, Integer> totaltype = totalsMap.get(type);
                
                int total = bv.getKoop1() + bv.getKoop2() + bv.getKoop3() + bv.getKoop4() + bv.getKooponb() + bv.getHuur1() + bv.getHuur2() + bv.getHuur3() + bv.getHuur4() + bv.getHuuronb() + bv.getOnbekend();
           
                totaltype.put("koop1", totaltype.get("koop1") + bv.getKoop1());
                totaltype.put("koop2", totaltype.get("koop2") + bv.getKoop2());
                totaltype.put("koop3", totaltype.get("koop3") + bv.getKoop3());
                totaltype.put("koop4", totaltype.get("koop4") + bv.getKoop4());
                totaltype.put("kooponb", totaltype.get("kooponb") + bv.getKooponb());
                totaltype.put("huur1", totaltype.get("huur1") + bv.getHuur1());
                totaltype.put("huur2", totaltype.get("huur2") + bv.getHuur2());
                totaltype.put("huur3", totaltype.get("huur3") + bv.getHuur3());
                totaltype.put("huur4", totaltype.get("huur4") + bv.getHuur4());
                totaltype.put("huuronb", totaltype.get("huuronb") + bv.getHuuronb());
                totaltype.put("onbekend", totaltype.get("onbekend") + bv.getOnbekend());
                totaltype.put("totaal", totaltype.get("totaal") + total);
                totaltype.put("subtotaal" + bv.getYear(), total);
            }
            Map<String, Integer> total = new HashMap<>();
            total.put("koop1", 0);
            total.put("koop2", 0);
            total.put("koop3", 0);
            total.put("koop4", 0);
            total.put("kooponb", 0);
            total.put("huur1", 0);
            total.put("huur2", 0);
            total.put("huur3", 0);
            total.put("huur4", 0);
            total.put("huuronb", 0);
            total.put("onbekend", 0);
            total.put("totaal", 0);
            
            for (String type : totalsMap.keySet()) {
                if(!types.containsKey(type)){
                    continue;
                }
                Map<String,Integer> cols = totalsMap.get(type);
                for (String col : total.keySet()) {
                    total.put(col, total.get(col) + cols.get(col));
                }
            }
            planning.put("bouwplanvariableTotalsNieuw", totalsMap);
            planning.put("totals", total);
        
        }
        planning.put("bouwplanvariables", bvArray);
        planning.put("types", typeInfo);
        return planning;
    }
    
    private JSONObject getOptionalVariables(Bouwplan bouwplan){
        EntityManager em = Stripersist.getEntityManager();
        JSONObject optionalVars = new JSONObject();
        Gemeente gemeente = bouwplan.getGemeente();
        Provincie p = gemeente.getProvincie();
        
        List<ConfigurableVariable> optionalLabelVars = em.createQuery("FROM ConfigurableVariable WHERE type = :type", ConfigurableVariable.class)
                .setParameter("type", VariableType.OPTIONAL).getResultList();

        List<ProvincieVariable> optVars = null;
        if (optionalLabelVars.size() > 0) {
            optVars = em.createQuery("FROM ProvincieVariable WHERE variable in :vars and provincie = :provincie")
                    .setParameter("vars", optionalLabelVars).setParameter("provincie", p).getResultList();
        }else{
            optVars = new ArrayList<ProvincieVariable>();
        }
        for (ProvincieVariable optVar : optVars) {
            optionalVars.put(optVar.getVariable().getName(), true);
        }
        return optionalVars;
    }
    
    private JSONObject  getSubcategories(Bouwplan bouwplan ) throws JSONException{
        EntityManager em = Stripersist.getEntityManager();
        List<ProvincieAdditionalVariable> vars = null;
        Gemeente gemeente = bouwplan.getGemeente();
        Provincie p = gemeente.getProvincie();
        if (gemeente.getRegio() != null) {
            vars = em.createQuery("select p FROM ProvincieAdditionalVariable p join p.regios r where p.label = false and p.mainCategory is not null and r.id = :regio order by additional_variable", ProvincieAdditionalVariable.class).setParameter("regio", gemeente.getRegio().getId()).getResultList();
        } else {
            vars = em.createQuery("FROM ProvincieAdditionalVariable where label = false and mainCategory is not null and provincie = :provincie and usedByMunicipalities = true order by additional_variable", ProvincieAdditionalVariable.class).setParameter("provincie", p).getResultList();
        }
        JSONObject subcategories = new JSONObject();
        for (ProvincieAdditionalVariable var : vars) {
            String location = var.getLocation();
            if(!subcategories.has(location)){
                JSONArray loc = new JSONArray();
                subcategories.put(location, loc);
            }
            JSONArray locationArray = subcategories.getJSONArray(location);
            JSONObject jsonVar = new JSONObject();
            jsonVar.put("var", var.getAdditional_variable() );
            jsonVar.put("maincategory", var.getMainCategory());
            locationArray.put(jsonVar);
            try{
                BouwplanAdditionalVariable bav = em.createQuery("FROM BouwplanAdditionalVariable WHERE bouwplan = :bouwplan and variable = :var", BouwplanAdditionalVariable.class).setParameter("bouwplan", bouwplan).setParameter("var", var).getSingleResult();

                List<ProvincieAdditionalVariableValue> values =  var.getValues();
                ProvincieAdditionalVariableValue value = null;
                for (ProvincieAdditionalVariableValue val : values) {
                    if(Integer.parseInt(bav.getVariableValue()) == val.getId()){
                        value = val;
                        break;
                    }
                }
                if(value != null){
                    jsonVar.put("value", value.getValue());
                }else{
                    jsonVar.put("value", bav.getVariableValue());
                }
            }catch(NoResultException ex ){
                
            }
        }
        return subcategories;
     }

    private JSONObject  getAdditionalVars(Bouwplan bouwplan ) throws JSONException{
        EntityManager em = Stripersist.getEntityManager();
        List<ProvincieAdditionalVariable> vars = null;
        Gemeente gemeente = bouwplan.getGemeente();
        Provincie p = gemeente.getProvincie();
        if (gemeente.getRegio() != null) {
            vars = em.createQuery("select p FROM ProvincieAdditionalVariable p join p.regios r where p.label = false and p.mainCategory is null and r.id = :regio order by additional_variable", ProvincieAdditionalVariable.class).setParameter("regio", gemeente.getRegio().getId()).getResultList();
        } else {
            vars = em.createQuery("FROM ProvincieAdditionalVariable where label = false and mainCategory is null and provincie = :provincie and usedByMunicipalities = true order by additional_variable", ProvincieAdditionalVariable.class).setParameter("provincie", p).getResultList();
        }

        JSONObject additionalVars = new JSONObject();
        for (ProvincieAdditionalVariable var : vars) {
            String location = var.getLocation();
            if(!additionalVars.has(location)){
                JSONArray loc = new JSONArray();
                additionalVars.put(location, loc);
            }
            JSONArray locationArray = additionalVars.getJSONArray(location);
            JSONObject jsonVar = new JSONObject();
            jsonVar.put("var", var.getAdditional_variable() );
            locationArray.put(jsonVar);
            try{
                BouwplanAdditionalVariable bav = em.createQuery("FROM BouwplanAdditionalVariable WHERE bouwplan = :bouwplan and variable = :var", BouwplanAdditionalVariable.class).setParameter("bouwplan", bouwplan).setParameter("var", var).getSingleResult();

                List<ProvincieAdditionalVariableValue> values =  var.getValues();
                ProvincieAdditionalVariableValue value = null;
                for (ProvincieAdditionalVariableValue val : values) {
                    if(Integer.parseInt(bav.getVariableValue()) == val.getId()){
                        value = val;
                        break;
                    }
                }
                if(value != null){
                    jsonVar.put("value", value.getValue());
                }else{
                    jsonVar.put("value", bav.getVariableValue());
                }
            }catch(NoResultException ex ){
                
            }
        }

        List<String> optionalLabelVars = em.createQuery("SELECT label FROM ConfigurableVariable WHERE type = :type", String.class)
                .setParameter("type", VariableType.LABEL).getResultList();

        List<ProvincieAdditionalVariable> labelVars = null;
        if (optionalLabelVars.size() > 0) {
            labelVars = em.createQuery("FROM ProvincieAdditionalVariable WHERE label = true and additional_variable in :vars and provincie = :provincie")
                    .setParameter("vars", optionalLabelVars).setParameter("provincie", p).getResultList();
        }else{
            labelVars = new ArrayList<ProvincieAdditionalVariable>();
        }
        Map<String,String> labelMap = new HashMap<String,String>();
        Regio regio = gemeente.getRegio();
        for (String optionalLabelVar : optionalLabelVars) {
            String value = null;
            for (ProvincieAdditionalVariable labelVar : labelVars) {
                if(labelVar.getAdditional_variable().equals(optionalLabelVar)){
                    for (ProvincieAdditionalVariableValue pavv : labelVar.getValues()) {
                        if(regio == null && pavv.isUsedByMunicipalities()){
                            value = pavv.getValue();
                        }
                        if(pavv.getRegios().contains(gemeente.getRegio())){
                            value = pavv.getValue();
                        }
                    }
                }
            }
            labelMap.put("" + optionalLabelVar, value != null ? value : optionalLabelVar);
        }

        JSONObject labels = new JSONObject(labelMap);
        additionalVars.put("labels",labels);
        return additionalVars;
    }

    /**
     * Get the image url from the CombineImageAction
     * @param param the json as string with params needed to create the image
     * @return url to (combined)image.
     */
    private String getImageUrl(Bouwplan bouwplan, int width, int height) throws Exception {
        RedirectResolution cia = new RedirectResolution(EditActionBean.class).addParameter("bouwplan", bouwplan.getId()).addParameter("width", width).addParameter("height", height).addParameter("bouwplanImage", Boolean.TRUE);
        String url= context.getRequest().getRequestURL().toString();
        String pm = "planmonitor/";
        int index = url.indexOf(pm);
        url = url.substring(0, index + pm.length());
        url += cia.getUrl(new Locale("NL"));
        return url;
    }

    // 2014, Eddy Scheper, ARIS B.V. - A5 and A0 added.
    private String getTemplateName(String pageFormat, String orientation) {
        if (A5.equalsIgnoreCase(pageFormat) && LANDSCAPE.equalsIgnoreCase(orientation)){
            return A5_Landscape;
        }else if (A5.equalsIgnoreCase(pageFormat) && PORTRAIT.equalsIgnoreCase(orientation)){
            return A5_Portrait;
        }else if (A4.equalsIgnoreCase(pageFormat) && LANDSCAPE.equalsIgnoreCase(orientation)){
            return A4_Landscape;
        }else if (A3.equalsIgnoreCase(pageFormat) && PORTRAIT.equalsIgnoreCase(orientation)){
            return A3_Portrait;
        }else if (A3.equalsIgnoreCase(pageFormat) && LANDSCAPE.equalsIgnoreCase(orientation)){
            return A3_Landscape;
        }else if (A0.equalsIgnoreCase(pageFormat) && PORTRAIT.equalsIgnoreCase(orientation)){
            return A0_Portrait;
        }else if (A0.equalsIgnoreCase(pageFormat) && LANDSCAPE.equalsIgnoreCase(orientation)){
            return A0_Landscape;
        }else{
            return A4_Portrait;
        }
    }
    //<editor-fold defaultstate="collapsed" desc="Getters and Setters">

    public List<Integer> getBouwplans() {
        return bouwplans;
    }

    public void setBouwplans(List<Integer> bouwplans) {
        this.bouwplans = bouwplans;
    }


    //</editor-fold>

    
    private boolean isAuthorized(){
        List<Bouwplan> newPlans = new ArrayList<Bouwplan>();
        final EntityManager em = Stripersist.getEntityManager();
        for (Integer id : bouwplans) {
            Bouwplan bouwplan = em.find(Bouwplan.class, id);
            if(BouwplanUtils.isAuthorized(bouwplan, gebruiker)){
                newPlans.add(bouwplan);
            }
        }
        if(newPlans.isEmpty()){
            return false;
        }else{
            bouwplans = new ArrayList<Integer>();
            for (Bouwplan plan : newPlans) {
                bouwplans.add(plan.getId());
            }
            return true;
        }
    }

    private String printInfoToString(PrintInfo info) throws JAXBException {
        JAXBContext context = JAXBContext.newInstance(PrintInfo.class);
        Marshaller m = context.createMarshaller();
        m.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, Boolean.TRUE);
        StringWriter sw = new StringWriter();
        m.marshal(info, sw);
        String s=sw.toString();
        return s;
    }

    private JSONArray getExtra(Bouwplan bouwplan) throws JSONException{
        JSONArray ar = new JSONArray();
        ar.put(getBasisgegevens(bouwplan));
        ar.put(getSoort(bouwplan));
        ar.put(getVoortgang(bouwplan));
        ar.put(getAantallen(bouwplan));
        ar.put(getWoningtypen(bouwplan));
        ar.put(getKoop(bouwplan));
        ar.put(getHuur(bouwplan));
        ar.put(getOverig(bouwplan));
        ar.put(getSloop(bouwplan));
        return ar;
    }


    private JSONObject getBasisgegevens(Bouwplan bouwplan) throws JSONException{
        JSONObject json = new JSONObject();
        json.put("className", "basisgegevens");
        json.put("componentName", "Basisgegevens");
        json.put("PlanID", bouwplan.getId());
        json.put("Plannaam", bouwplan.getNaam());
        json.put("DatumLaatsteWijziging",  df.format(bouwplan.getDatumlaatstewijziging()));
        json.put("Gemeente", bouwplan.getGemeente().getNaam());
        json.put("Referentienummer", bouwplan.getReferentienummer());
        json.put("Bestemmingsplan", bouwplan.getBestemmingsplannaam());
        json.put("IMRO-nummer", bouwplan.getImronummer());
        json.put("Masterplan", bouwplan.getMasterplan());
        if(bouwplan.getOntwikkelaartype() != null){
            json.put("Oprachtgevertype", bouwplan.getOntwikkelaartype().getType());
        }
        json.put("Opdrachtgever", bouwplan.getOntwikkelaar_namelijk());
        json.put("Vertrouwelijk", bouwplan.getGebruikgegevens().getNaam());
        return json;
    }

    private JSONObject getSoort(Bouwplan bouwplan)throws JSONException{
        JSONObject json = new JSONObject();
        json.put("className", "Soortplan");
        json.put("componentName", "Soortplan");
        json.put("Plantype", bouwplan.getPlantype().getType());
        if(bouwplan.getWoonmilieu() != null){
            json.put("Woonmilieu", bouwplan.getWoonmilieu().getType());
        }
        json.put("Woonmilieuabf14", bouwplan.getWoonmilieuabf14());
        return json;
    }

    private JSONObject getVoortgang(Bouwplan bouwplan)throws JSONException{
        JSONObject json = new JSONObject();
        json.put("className", "Voortgang");
        json.put("componentName", "Voortgang");
        json.put("Statusplanologisch", bouwplan.getStatusplanologisch().getType());
        json.put("Statusproject", bouwplan.getStatusproject().getType());
        json.put("Knelpunten", bouwplan.getKnelpunten());
        return json;
    }
    
    private JSONObject getSloop(Bouwplan bouwplan){
        JSONObject json = new JSONObject();
        json.put("className", "Sloop");
        json.put("componentName", "Sloop");
        json.put("resterendesloop", bouwplan.getResterendesloop());
        json.put("totaalsloop", bouwplan.getTotaalsloop());
        json.put("gerealiseerdesloop", bouwplan.getGerealiseerdesloop());
        return json;
    }

    private JSONObject getAantallen(Bouwplan bouwplan)throws JSONException{
        JSONObject json = new JSONObject();
        json.put("className", "Aantallen");
        json.put("componentName", "Aantallen");
        json.put("eersteoplevering", bouwplan.getJaar_eerste_oplevering());
        json.put("laatsteoplevering", bouwplan.getJaar_laatste_oplevering());
        json.put("restcapaciteit", bouwplan.getRestcapaciteit());
        json.put("totaalcapaciteit", bouwplan.getTotaalcapaciteit());
        json.put("gerealiseerdecapaciteit", bouwplan.getGerealiseerdecapaciteit());
        json.put("resterendesloop", bouwplan.getResterendesloop());
        return json;
    }

    private JSONObject getWoningtypen(Bouwplan bouwplan)throws JSONException{
        JSONObject json = new JSONObject();
        json.put("className", "Woningtypen");
        json.put("componentName", "Woningtypen");
        json.put("Meergezins", bouwplan.getMeergezins());
        json.put("Eengezins", bouwplan.getEensgezins());
        json.put("Woningtypeonbekend", bouwplan.getWoningtypeonbekend());
        return json;
    }

    private JSONObject getKoop(Bouwplan bouwplan)throws JSONException{
        JSONObject json = new JSONObject();
        json.put("className", "Koop");
        json.put("componentName", "Koop");
        json.put("Koop1", bouwplan.getKoop1());
        json.put("Koop2", bouwplan.getKoop2());
        json.put("Koop3", bouwplan.getKoop3());
        json.put("Koop4", bouwplan.getKoop4());
        json.put("Kooponbekend", bouwplan.getKoopprijsonbekend());
        json.put("Kooptotaal", bouwplan.getKooptotaal());
        json.put("OnbekendOnbekend", bouwplan.getOnbekendonbekend());
        return json;
    }

    private JSONObject getHuur(Bouwplan bouwplan)throws JSONException{
        JSONObject json = new JSONObject();
        json.put("className", "Huur");
        json.put("componentName", "Huur");
        json.put("Huur1", bouwplan.getHuur1());
        json.put("Huur2", bouwplan.getHuur2());
        json.put("Huur3", bouwplan.getHuur3());
        json.put("Huur4", bouwplan.getHuur4());
        json.put("Huuronbekend", bouwplan.getHuurprijsonbekend());
        json.put("Huurtotaal", bouwplan.getHuurtotaal());
        return json;
    }

    private JSONObject getOverig(Bouwplan bouwplan)throws JSONException{
        JSONObject json = new JSONObject();
        json.put("className", "Overig");
        json.put("componentName", "Overig");
        if(!bouwplan.isOpmerkingenprive()){
            json.put("opmerking",bouwplan.getOpmerkingen() );
        }
        return json;
    }
    
}
