/*
 * Decompiled with CFR 0.152.
 */
package org.tailormap.api.controller;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.micrometer.core.annotation.Counted;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.validation.ConstraintViolationException;
import jakarta.validation.constraints.Email;
import jakarta.validation.constraints.NotNull;
import java.io.Serializable;
import java.lang.invoke.MethodHandles;
import java.time.Instant;
import java.time.ZoneId;
import java.util.Locale;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.RejectedExecutionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.MessageSource;
import org.springframework.http.HttpStatus;
import org.springframework.http.HttpStatusCode;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.mail.MailException;
import org.springframework.mail.SimpleMailMessage;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.server.ResponseStatusException;
import org.springframework.web.servlet.LocaleResolver;
import org.tailormap.api.configuration.TailormapPasswordStrengthConfig;
import org.tailormap.api.persistence.TemporaryToken;
import org.tailormap.api.persistence.User;
import org.tailormap.api.repository.TemporaryTokenRepository;
import org.tailormap.api.repository.UserRepository;
import org.tailormap.api.util.TMPasswordDeserializer;
import org.tailormap.api.viewer.model.ErrorResponse;

@RestController
@Validated
@ConditionalOnProperty(name={"tailormap-api.password-reset.enabled"}, havingValue="true")
public class PasswordResetController {
    private static final Logger logger = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private final JavaMailSender emailSender;
    private final UserRepository userRepository;
    private final TemporaryTokenRepository temporaryTokenRepository;
    private final MessageSource messageSource;
    private final LocaleResolver localeResolver;
    @Value(value="${tailormap-api.mail.from}")
    private String mailFrom;
    @Value(value="${tailormap-api.password-reset.enabled:false}")
    private boolean passwordResetEnabled;
    @Value(value="${tailormap-api.password-reset.disabled-for}")
    private Set<String> passwordResetDisabledFor;
    @Value(value="${tailormap-api.password-reset.token-expiration-minutes:5}")
    private int passwordResetTokenExpirationMinutes;

    public PasswordResetController(JavaMailSender emailSender, UserRepository userRepository, TemporaryTokenRepository temporaryTokenRepository, MessageSource messageSource, LocaleResolver localeResolver) {
        this.emailSender = emailSender;
        this.userRepository = userRepository;
        this.temporaryTokenRepository = temporaryTokenRepository;
        this.messageSource = messageSource;
        this.localeResolver = localeResolver;
    }

    @ExceptionHandler(value={ConstraintViolationException.class})
    public ResponseEntity<?> handleException(ConstraintViolationException e) {
        return ResponseEntity.status((HttpStatusCode)HttpStatus.BAD_REQUEST).contentType(MediaType.APPLICATION_JSON).body((Object)new ErrorResponse().message(e.getMessage()).code(Integer.valueOf(HttpStatus.BAD_REQUEST.value())));
    }

    @PostMapping(path={"${tailormap-api.base-path}/password-reset"}, produces={"application/json"}, consumes={"application/x-www-form-urlencoded"})
    @Counted(value="tailormap_api_password_reset_request", description="number of password reset requests by email")
    public ResponseEntity<Serializable> requestPasswordReset(@RequestParam @Email String email, HttpServletRequest request) throws ConstraintViolationException {
        if (this.passwordResetEnabled && !this.passwordResetDisabledFor.contains(email)) {
            this.sendPasswordResetEmail(email, request);
        }
        return ResponseEntity.accepted().body((Object)new ObjectMapper().createObjectNode().put("message", "Your password reset request is being processed"));
    }

    @PostMapping(path={"${tailormap-api.base-path}/user/reset-password"}, produces={"application/json"}, consumes={"application/x-www-form-urlencoded"})
    @Counted(value="tailormap_api_password_reset_confirmation", description="number of submitted password reset confirmations")
    @Transactional
    public ResponseEntity<Serializable> confirmPasswordReset(@NotNull UUID token, @NotNull String username, @NotNull String newPassword) throws ResponseStatusException {
        User user;
        TemporaryToken temporaryToken = (TemporaryToken)this.temporaryTokenRepository.findById((Object)token).orElseThrow(() -> new ResponseStatusException((HttpStatusCode)HttpStatus.BAD_REQUEST, "Token invalid"));
        boolean validation = TailormapPasswordStrengthConfig.getValidation();
        int minLength = TailormapPasswordStrengthConfig.getMinLength();
        int minStrength = TailormapPasswordStrengthConfig.getMinStrength();
        if (validation && !TMPasswordDeserializer.validatePasswordStrength((String)newPassword, (int)minLength, (int)minStrength)) {
            throw new ResponseStatusException((HttpStatusCode)HttpStatus.BAD_REQUEST, "Password too short or too easily guessable");
        }
        if (temporaryToken.getExpirationTime().isAfter(Instant.now().atZone(ZoneId.of("UTC"))) && temporaryToken.getTokenType() == TemporaryToken.TokenType.PASSWORD_RESET && temporaryToken.getUsername().equals(username) && (user = (User)this.userRepository.findById(username).orElse(null)) != null && user.isEnabledAndValidUntil()) {
            this.userRepository.updatePassword(username, TMPasswordDeserializer.encoder().encode((CharSequence)newPassword));
            logger.info("Password reset successful for user: {}", (Object)user.getUsername());
            this.temporaryTokenRepository.delete((Object)temporaryToken);
            return ResponseEntity.status((HttpStatusCode)HttpStatus.OK).body((Object)new ObjectMapper().createObjectNode().put("message", "Your password reset was reset successful"));
        }
        throw new ResponseStatusException((HttpStatusCode)HttpStatus.BAD_REQUEST, "Token expired or invalid request");
    }

    private void sendPasswordResetEmail(String email, HttpServletRequest request) {
        String absoluteLinkPrefix = request.getRequestURL().toString().replace(request.getRequestURI(), request.getContextPath());
        Locale locale = this.localeResolver.resolveLocale(request);
        try (ExecutorService emailExecutor = Executors.newSingleThreadExecutor();){
            emailExecutor.execute(() -> {
                try {
                    this.userRepository.findByEmail(email).ifPresent(user -> {
                        if (!user.isEnabledAndValidUntil()) {
                            return;
                        }
                        TemporaryToken token = new TemporaryToken(TemporaryToken.TokenType.PASSWORD_RESET, user.getUsername(), this.passwordResetTokenExpirationMinutes);
                        token = (TemporaryToken)this.temporaryTokenRepository.save((Object)token);
                        String absoluteLink = absoluteLinkPrefix + "/user/password-reset/" + token.getCombinedTokenAndExpirationAsBase64();
                        SimpleMailMessage message = new SimpleMailMessage();
                        message.setFrom(this.mailFrom);
                        message.setTo(user.getEmail());
                        message.setSubject(this.messageSource.getMessage("reset-password-request.email-subject", null, locale));
                        message.setText(this.messageSource.getMessage("reset-password-request.email-body", new Object[]{absoluteLink}, locale));
                        logger.trace("Sending message {}", (Object)message);
                        logger.info("Sending password reset email for user: {}", (Object)user.getUsername());
                        this.emailSender.send(message);
                    });
                }
                catch (MailException e) {
                    logger.error("Failed to send password reset email", (Throwable)e);
                }
                catch (Exception e) {
                    logger.error("Unexpected exception in password reset email thread", (Throwable)e);
                }
            });
            emailExecutor.shutdown();
        }
        catch (RejectedExecutionException e) {
            logger.error("Failed to start password reset email thread", (Throwable)e);
        }
    }
}

