package nl.tailormap.viewer.userlayer;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.List;

public class PostgreSQL implements DataBase {
    private static final Log LOG = LogFactory.getLog(PostgreSQL.class);
    private static final String SELECT_SQL = " SELECT * FROM %s %s ";
    private static final String CREATE_SQL = "CREATE OR REPLACE VIEW %s AS " + SELECT_SQL;
    private static final String DROP_SQL = "DROP VIEW IF EXISTS %s";
    private static final String COMMENTS_SQL = "COMMENT ON VIEW %s IS '%s'";
    private static final String INSERT_GT_PK_METADATA = "INSERT INTO gt_pk_metadata (table_schema, table_name, pk_column, " +
            "pk_column_idx, pk_policy) values ('%s','%s', '%s', %d, 'autogenerated')";
    private static final String REMOVE_GT_PK_METADATA = "DELETE FROM gt_pk_metadata where table_schema = '%s' " +
            "and table_name = '%s'";

    private final Connection connection;

    private final String GT_PK_METADATA_DDL = "CREATE TABLE gt_pk_metadata (" +
            "  table_schema VARCHAR(32) NOT NULL," +
            "  table_name VARCHAR(120) NOT NULL," +
            "  pk_column VARCHAR(32) NOT NULL," +
            "  pk_column_idx INTEGER," +
            "  pk_policy VARCHAR(32)," +
            "  pk_sequence VARCHAR(64)," +
            "  unique (table_schema, table_name, pk_column)," +
            "  check (pk_policy in ('sequence', 'assigned', 'autogenerated')))";

    public PostgreSQL(Connection connection) {
        this.connection = connection;
    }

    @Override
    public String preValidateView(String tableName, String filterSQL) {
        String result = null;
        try (PreparedStatement ps = connection.prepareStatement(
                String.format(SELECT_SQL, tableName, filterSQL))
        ) {
            ps.setFetchSize(1);
            ps.setMaxRows(1);
            ps.execute();
        } catch (SQLException throwables) {
            LOG.error(throwables.getLocalizedMessage());
            result = String.format(INVALID_MSG, throwables.getLocalizedMessage()).replaceAll("\n", " ");
        }
        return result;
    }

    @Override
    public String getGtPkMetadataDDL() {
        return GT_PK_METADATA_DDL;
    }

    /**
     *
     * Note: <strong>comments are not sanitized!</strong> see inline comments.
     *
     * @param viewName  Name of the view
     * @param tableName Name of the source table
     * @param filterSQL Filter definition of view (where clause)
     * @param comments  optional comments to add the the view, can be {@code null}
     * @return {@code true} if call succeeded
     */
    @Override
    public boolean createView(String viewName, String tableName, String filterSQL, String comments) {
        boolean result;
        LOG.debug("try to create view " + viewName + " using table " + tableName + " and query " + filterSQL);
        try (PreparedStatement ps = connection.prepareStatement(
                String.format(CREATE_SQL, viewName, tableName, filterSQL));
                /* we don't need to sanitize the comments as it does not actually contain any user input.
                 * ans since this is a DDL statement there is no parameter binding and the statement would
                 * already be executed anyway during prepare
                 */
             PreparedStatement psComment = connection.prepareStatement(
                     String.format(COMMENTS_SQL, viewName, comments.replaceAll("'", "''")))
        ) {
            // we will ignore any results of setting the comment, only the view creating is important
            result = (0 == ps.executeUpdate());
            psComment.executeUpdate();
            if(result){
                addToGtPKMetadata(viewName, tableName);
            }
        } catch (SQLException throwables) {
            LOG.error("Probleem tijdens maken van view: " + throwables.getLocalizedMessage());
            result = false;
        }
        return result;
    }

    @Override
    public boolean dropView(String viewName) {
        boolean result;
        removeFromGtPKMetadata(viewName);
        LOG.debug("try to drop view " + viewName);
        try (PreparedStatement ps = connection.prepareStatement(String.format(DROP_SQL, viewName))) {
            result = (0 == ps.executeUpdate());
        } catch (SQLException throwables) {
            LOG.error("Probleem tijdens droppen van view: " + throwables.getLocalizedMessage());
            result = false;
        }

        return result;
    }

    @Override
    public boolean addToGtPKMetadata(String viewName, String table) {
        try {
            if(!doesGtPkMetadataExists(this.connection)){
                createGtPkMetadata(this.connection);
            }
        } catch (SQLException throwables) {
            LOG.error("Cannot check if gt_pk_metadata exists or create it: "+ throwables.getLocalizedMessage());
        }
        boolean result = false;
        LOG.debug("try to insert view " + viewName + " into gt_pk_metadata");
        try {
            List<String> schemas = this.getSchema(table, this.connection);
            if(schemas.isEmpty()){
                LOG.error("Cannot add view to gt_pk_metadatatable. Cannot retrieve schema for " + table);
                return false;
            }
            String schema = schemas.get(0);
            List<String> pks =this.getPrimaryKey(table, schema, this.connection);
            for (int i = 0; i < pks.size(); i++) {
                String pkColumn = pks.get(i);
                try (PreparedStatement ps = connection.prepareStatement(
                        //(table_schema, table_name, pk_column, pk_column_idx, pk_policy)
                        String.format(INSERT_GT_PK_METADATA, schema, viewName, pkColumn, i));
                ) {
                    result = (1 == ps.executeUpdate());
                } catch (SQLException throwables) {
                    LOG.error("Probleem tijdens uitvoeren insert: " + String.format(INSERT_GT_PK_METADATA, schema, viewName, pkColumn, i) + ", with error:" + throwables.getLocalizedMessage());
                }
            }
        } catch (Exception throwables) {
            LOG.error("Probleem tijdens ophalen PK kolom of schemanaam: " + throwables.getLocalizedMessage());
            result = false;
        }
        return result;
    }

    @Override
    public boolean removeFromGtPKMetadata(String viewName) {
        boolean result = false;
        LOG.debug("try to remove view " + viewName + " from gt_pk_metadata");
        try {
            List<String> schemas = this.getSchema(viewName, this.connection);
            if (schemas.isEmpty()) {
                LOG.error("Cannot remove view from gt_pk_metadatatable. Cannot retrieve schema for " + viewName);
                return false;
            }
            String schema = schemas.get(0);
            try (PreparedStatement ps = connection.prepareStatement(
                    String.format(REMOVE_GT_PK_METADATA, schema, viewName));
            ) {
                result = (1 == ps.executeUpdate());
            } catch (SQLException throwables) {
                LOG.error("Probleem tijdens uitvoeren delete: " + String.format(REMOVE_GT_PK_METADATA, schema, viewName) + ", with error:" + throwables.getLocalizedMessage());
            }
        } catch (Exception throwables) {
            LOG.error("Probleem tijdens ophalen schemanaam: " + throwables.getLocalizedMessage());
            result = false;
        }
        return result;
    }

    /**
     * close any resources.
     */
    @Override
    public void close() {
        try {
            this.connection.close();
        } catch (SQLException throwables) {
            LOG.error("Probleem tijdens sluiten van connectie: " + throwables.getLocalizedMessage());
        }
    }
}
